package com.example.tester

import android.content.Context
import android.os.Bundle
import androidx.activity.ComponentActivity
import androidx.activity.compose.setContent
import androidx.compose.foundation.clickable
import androidx.compose.foundation.layout.*
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.*
import androidx.compose.material3.*
import androidx.compose.runtime.*
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import androidx.compose.ui.graphics.RectangleShape // Para botones cuadrados/simples
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.text.font.FontWeight
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextAlign
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.navigation.NavHostController
import androidx.navigation.compose.NavHost
import androidx.navigation.compose.composable
import androidx.navigation.compose.rememberNavController
import com.android.volley.Request
import com.android.volley.toolbox.JsonArrayRequest
import com.android.volley.toolbox.StringRequest
import com.android.volley.toolbox.Volley
import org.json.JSONObject

// ==========================================
// 1. MAIN ACTIVITY
// ==========================================

class MainActivity : ComponentActivity() {
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContent {
            // Tema con colores profesionales y limpios
            MaterialTheme(
                colorScheme = lightColorScheme(
                    primary = Color(0xFF2C3E50),      // Azul oscuro corporativo
                    onPrimary = Color.White,
                    secondary = Color(0xFF95A5A6),    // Gris elegante
                    tertiary = Color(0xFF1ABC9C),     // Turquesa sutil
                    error = Color(0xFFE74C3C),        // Rojo plano
                    surface = Color(0xFFF8F9F9)       // Fondo casi blanco
                )
            ) {
                PantallaNavegacionPrincipal()
            }
        }
    }
}

// ==========================================
// 2. DATA CLASSES
// ==========================================

data class Reserva(
    val numeroHabitacion: Int,
    val nombreHuesped: String,
    val fechaEntrada: String,
    val fechaSalida: String,
    val precioNoche: Float,
    val tipoHabitacion: String,
    val estado: String = "Disponible" // Agregamos estado para control interno
)

data class HabitacionEstado(
    val numeroHabitacion: String,
    val nombreHuesped: String?,
    val estado: String
)

// ==========================================
// 3. ESTRUCTURA DE NAVEGACIÓN
// ==========================================

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun PantallaNavegacionPrincipal() {
    val navController = rememberNavController()
    var selectedScreen by remember { mutableStateOf("lista") }

    Scaffold(
        topBar = {
            // TopBar minimalista sin sombras fuertes
            CenterAlignedTopAppBar(
                title = { Text("APP HOTEL", fontWeight = FontWeight.Bold, letterSpacing = 1.sp, textAlign = TextAlign.Center, modifier = Modifier.fillMaxWidth()) },
                colors = TopAppBarDefaults.centerAlignedTopAppBarColors(
                    containerColor = MaterialTheme.colorScheme.primary,
                    titleContentColor = MaterialTheme.colorScheme.onPrimary
                )
            )
        },
        bottomBar = {
            NavigationBar(containerColor = Color.White) {
                NavigationBarItem(
                    selected = selectedScreen == "lista",
                    onClick = { selectedScreen = "lista"; navController.navigate("pantalla_lista") { popUpTo(0) } },
                    icon = { Icon(Icons.Filled.Bed, null) },
                    label = { Text("Habitaciones") }
                )
                NavigationBarItem(
                    selected = selectedScreen == "formulario",
                    onClick = { selectedScreen = "formulario"; navController.navigate("pantalla_formulario") },
                    icon = { Icon(Icons.Filled.EditCalendar, null) },
                    label = { Text("Gestión") }
                )
                NavigationBarItem(
                    selected = selectedScreen == "reporte",
                    onClick = { selectedScreen = "reporte"; navController.navigate("pantalla_reporte") },
                    icon = { Icon(Icons.Filled.Analytics, null) },
                    label = { Text("Reporte") }
                )
            }
        }
    ) { paddingValues ->
        NavHost(navController = navController, startDestination = "pantalla_lista", modifier = Modifier.padding(paddingValues)) {
            composable("pantalla_lista") { PantallaListaHabitaciones() }
            composable("pantalla_formulario") { PantallaFormularioCRUD() }
            composable("pantalla_reporte") { PantallaReporte() }
        }
    }
}

// ==========================================
// 4. PANTALLA 1: LISTA POR TIPOS
// ==========================================

@Composable
fun PantallaListaHabitaciones() {
    val navController = rememberNavController()
    NavHost(navController = navController, startDestination = "menu_tipos") {
        composable("menu_tipos") { MenuTiposHabitacion(navController) }
        composable("lista_filtrada/{tipo}") { backStackEntry ->
            val tipo = backStackEntry.arguments?.getString("tipo") ?: ""
            ListaHabitacionesFiltrada(tipo, navController)
        }
    }
}

@Composable
fun MenuTiposHabitacion(navController: NavHostController) {
    val tipos = listOf("Privadas", "Matrimoniales", "Dobles", "Triple")
    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Text("CATEGORÍAS", style = MaterialTheme.typography.titleMedium, color = Color.Gray, modifier = Modifier.padding(bottom = 12.dp))
        LazyColumn(verticalArrangement = Arrangement.spacedBy(10.dp)) {
            items(tipos) { tipo ->
                Card(
                    modifier = Modifier.fillMaxWidth().height(80.dp).clickable { navController.navigate("lista_filtrada/$tipo") },
                    colors = CardDefaults.cardColors(containerColor = Color.White),
                    elevation = CardDefaults.cardElevation(1.dp), // Sombra mínima
                    shape = RoundedCornerShape(8.dp)
                ) {
                    Row(modifier = Modifier.fillMaxSize().padding(horizontal = 20.dp), verticalAlignment = Alignment.CenterVertically) {
                        Icon(Icons.Filled.BedroomParent, null, tint = MaterialTheme.colorScheme.primary)
                        Spacer(modifier = Modifier.width(16.dp))
                        Text(tipo, style = MaterialTheme.typography.titleMedium)
                        Spacer(modifier = Modifier.weight(1f))
                        Icon(Icons.Filled.ChevronRight, null, tint = Color.LightGray)
                    }
                }
            }
        }
    }
}

@Composable
fun ListaHabitacionesFiltrada(tipoHabitacion: String, navController: NavHostController) {
    val contexto = LocalContext.current
    var listaHabitaciones by remember { mutableStateOf(listOf<HabitacionEstado>()) }
    var cargando by remember { mutableStateOf(true) }

    LaunchedEffect(tipoHabitacion) {
        ConsultarHabitacionesPorTipo(contexto, tipoHabitacion) { lista ->
            listaHabitaciones = lista
            cargando = false
        }
    }

    Column(modifier = Modifier.fillMaxSize().padding(16.dp)) {
        Row(verticalAlignment = Alignment.CenterVertically, modifier = Modifier.padding(bottom = 16.dp)) {
            IconButton(onClick = { navController.popBackStack() }) { Icon(Icons.Filled.ArrowBack, null) }
            Text(tipoHabitacion, style = MaterialTheme.typography.headlineSmall)
        }
        if (cargando) Box(Modifier.fillMaxSize(), contentAlignment = Alignment.Center) { CircularProgressIndicator() }
        else LazyColumn(verticalArrangement = Arrangement.spacedBy(8.dp)) {
            items(listaHabitaciones) { hab ->
                val ocupada = hab.estado.equals("Ocupada", ignoreCase = true)
                Card(
                    colors = CardDefaults.cardColors(containerColor = if (ocupada) Color(0xFFFDEDEC) else Color(0xFFEAFAF1)), // Rojo/Verde muy pastel
                    modifier = Modifier.fillMaxWidth(),
                    shape = RoundedCornerShape(4.dp) // Bordes casi rectos
                ) {
                    Row(Modifier.padding(16.dp), verticalAlignment = Alignment.CenterVertically) {
                        Column {
                            Text("HABITACIÓN ${hab.numeroHabitacion}", fontWeight = FontWeight.Bold)
                            Text(if (ocupada) hab.nombreHuesped ?: "" else "Disponible", style = MaterialTheme.typography.bodySmall)
                        }
                        Spacer(Modifier.weight(1f))
                        Text(hab.estado, color = if (ocupada) Color(0xFFE74C3C) else Color(0xFF27AE60), fontWeight = FontWeight.Bold)
                    }
                }
            }
        }
    }
}

// ==========================================
// 5. PANTALLA GESTIÓN: "SUPER GOOD" Y SIMPLE
// ==========================================

@Composable
fun PantallaFormularioCRUD() {
    val contexto = LocalContext.current
    val scrollState = rememberScrollState()

    // Estados
    var numeroHabitacion by remember { mutableStateOf("") }
    var nombreHuesped by remember { mutableStateOf("") }
    var fechaEntrada by remember { mutableStateOf("") }
    var fechaSalida by remember { mutableStateOf("") }
    var precioNoche by remember { mutableStateOf("") }
    var tipoHabitacion by remember { mutableStateOf("") }

    // Control
    var mensaje by remember { mutableStateOf("Ingrese N° de habitación para comenzar") }
    var esEdicion by remember { mutableStateOf(false) } // true=Ocupada, false=Libre
    var botonesHabilitados by remember { mutableStateOf(false) }

    Column(
        modifier = Modifier.fillMaxSize().padding(24.dp).verticalScroll(scrollState),
        verticalArrangement = Arrangement.spacedBy(16.dp)
    ) {
        Text("GESTIÓN DE RESERVAS", style = MaterialTheme.typography.titleLarge, fontWeight = FontWeight.Bold, color = MaterialTheme.colorScheme.primary)

        // 1. BUSCADOR INTELIGENTE
        Row(verticalAlignment = Alignment.CenterVertically) {
            OutlinedTextField(
                value = numeroHabitacion,
                onValueChange = { numeroHabitacion = it },
                label = { Text("N° Habitación") },
                modifier = Modifier.weight(1f),
                keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
                singleLine = true
            )
            Spacer(Modifier.width(10.dp))
            // Botón de búsqueda simple y plano
            Button(
                onClick = {
                    if (numeroHabitacion.isNotEmpty()) {
                        ConsultaHabitacion(numeroHabitacion, { r ->
                            if (r != null) {
                                // AUTO-RELLENO DE DATOS
                                tipoHabitacion = r.tipoHabitacion
                                precioNoche = r.precioNoche.toString() // ¡PRECIO AUTOMÁTICO!
                                botonesHabilitados = true

                                if (r.estado == "Ocupada") {
                                    // MODO OCUPADA
                                    nombreHuesped = r.nombreHuesped
                                    fechaEntrada = r.fechaEntrada
                                    fechaSalida = r.fechaSalida
                                    esEdicion = true
                                    mensaje = "Habitación Ocupada. Datos cargados."
                                } else {
                                    // MODO DISPONIBLE
                                    nombreHuesped = ""
                                    fechaEntrada = ""
                                    fechaSalida = ""
                                    esEdicion = false
                                    mensaje = "Habitación Disponible. Precio: S/. $precioNoche"
                                }
                            } else {
                                mensaje = "Habitación no existe"
                                botonesHabilitados = false
                            }
                        }, contexto)
                    }
                },
                shape = RoundedCornerShape(4.dp),
                colors = ButtonDefaults.buttonColors(containerColor = MaterialTheme.colorScheme.primary),
                contentPadding = PaddingValues(horizontal = 20.dp, vertical = 15.dp)
            ) { Icon(Icons.Filled.Search, null) }
        }

        Text(mensaje, style = MaterialTheme.typography.bodyMedium, color = Color.Gray)

        Divider()

        // 2. FORMULARIO AUTOMATIZADO
        if (botonesHabilitados) {
            OutlinedTextField(
                value = tipoHabitacion, onValueChange = {}, readOnly = true,
                label = { Text("Categoría") }, modifier = Modifier.fillMaxWidth(),
                colors = OutlinedTextFieldDefaults.colors(unfocusedContainerColor = Color(0xFFF2F3F4))
            )

            // CAMPO DE PRECIO BLOQUEADO (AUTOMÁTICO)
            OutlinedTextField(
                value = precioNoche, onValueChange = {}, readOnly = true,
                label = { Text("Precio por Noche (Automático)") }, modifier = Modifier.fillMaxWidth(),
                leadingIcon = { Text("S/.", fontWeight = FontWeight.Bold) },
                colors = OutlinedTextFieldDefaults.colors(unfocusedContainerColor = Color(0xFFF2F3F4)) // Gris claro para indicar read-only
            )

            OutlinedTextField(
                value = nombreHuesped, onValueChange = { nombreHuesped = it },
                label = { Text("Nombre Completo") }, modifier = Modifier.fillMaxWidth()
            )

            Row(horizontalArrangement = Arrangement.spacedBy(10.dp)) {
                OutlinedTextField(
                    value = fechaEntrada, onValueChange = { fechaEntrada = it },
                    label = { Text("Entrada (AAAA-MM-DD)") }, modifier = Modifier.weight(1f)
                )
                OutlinedTextField(
                    value = fechaSalida, onValueChange = { fechaSalida = it },
                    label = { Text("Salida (AAAA-MM-DD)") }, modifier = Modifier.weight(1f)
                )
            }

            Spacer(Modifier.height(10.dp))

            // 3. BOTONES SIMPLES (SIN EFECTOS)
            if (!esEdicion) {
                // BOTÓN REGISTRAR (Verde Plano)
                Button(
                    onClick = {
                        if (nombreHuesped.isNotEmpty() && fechaEntrada.isNotEmpty()) {
                            val r = Reserva(numeroHabitacion.toInt(), nombreHuesped, fechaEntrada, fechaSalida, precioNoche.toFloat(), tipoHabitacion)
                            GuardarReserva(r, { if(it) { mensaje="Registrado"; esEdicion=true } else mensaje="Error" }, contexto)
                        }
                    },
                    modifier = Modifier.fillMaxWidth().height(50.dp),
                    shape = RoundedCornerShape(4.dp), // Casi cuadrado
                    colors = ButtonDefaults.buttonColors(containerColor = Color(0xFF27AE60)), // Verde simple
                    elevation = ButtonDefaults.buttonElevation(0.dp) // SIN SOMBRA
                ) { Text("REGISTRAR CHECK-IN", fontWeight = FontWeight.Bold) }
            } else {
                // BOTONES EDICIÓN (Planos)
                Row(horizontalArrangement = Arrangement.spacedBy(10.dp)) {
                    Button(
                        onClick = {
                            val r = Reserva(numeroHabitacion.toInt(), nombreHuesped, fechaEntrada, fechaSalida, precioNoche.toFloat(), tipoHabitacion)
                            ModificarReserva(r, { mensaje = if(it) "Actualizado" else "Error" }, contexto)
                        },
                        modifier = Modifier.weight(1f).height(50.dp),
                        shape = RoundedCornerShape(4.dp),
                        colors = ButtonDefaults.buttonColors(containerColor = Color(0xFF2980B9)), // Azul simple
                        elevation = ButtonDefaults.buttonElevation(0.dp)
                    ) { Text("GUARDAR CAMBIOS") }

                    Button(
                        onClick = {
                            BorrarReserva(numeroHabitacion, {
                                if(it) { mensaje="Habitación Liberada"; esEdicion=false; nombreHuesped=""; fechaEntrada=""; fechaSalida="" }
                            }, contexto)
                        },
                        modifier = Modifier.weight(1f).height(50.dp),
                        shape = RoundedCornerShape(4.dp),
                        colors = ButtonDefaults.buttonColors(containerColor = Color(0xFFC0392B)), // Rojo simple
                        elevation = ButtonDefaults.buttonElevation(0.dp)
                    ) { Text("CHECK-OUT") }
                }

                // BOTÓN PDF (Gris simple)
                Button(
                    onClick = {
                        val r = Reserva(numeroHabitacion.toInt(), nombreHuesped, fechaEntrada, fechaSalida, precioNoche.toFloat(), tipoHabitacion)
                        PDFManager.GenerarComprobantePDF(contexto, r)
                    },
                    modifier = Modifier.fillMaxWidth().padding(top = 10.dp),
                    shape = RoundedCornerShape(4.dp),
                    colors = ButtonDefaults.buttonColors(containerColor = Color(0xFF7F8C8D)), // Gris
                    elevation = ButtonDefaults.buttonElevation(0.dp)
                ) { Icon(Icons.Filled.Print, null); Spacer(Modifier.width(8.dp)); Text("IMPRIMIR COMPROBANTE") }
            }
        }
    }
}

// ==========================================
// 6. PANTALLA REPORTE
// ==========================================
@Composable
fun PantallaReporte() {
    val ctx = LocalContext.current
    var total by remember { mutableStateOf(0.0) }
    var ocupadas by remember { mutableStateOf(0) }

    LaunchedEffect(Unit) { ListarTodasLasReservas(ctx) { l -> ocupadas=l.size; total=l.sumOf { it.precioNoche.toDouble() } } }

    Column(Modifier.fillMaxSize(), horizontalAlignment = Alignment.CenterHorizontally, verticalArrangement = Arrangement.Center) {
        Text("INGRESOS DEL DÍA", style = MaterialTheme.typography.labelLarge, color = Color.Gray)
        Text("S/. $total", style = MaterialTheme.typography.displayLarge, fontWeight = FontWeight.Bold, color = MaterialTheme.colorScheme.primary)
        Spacer(Modifier.height(20.dp))
        Text("HABITACIONES OCUPADAS: $ocupadas", style = MaterialTheme.typography.titleLarge)
    }
}

// ==========================================
// 7. FUNCIONES DE RED (SIMPLIFICADAS)
// ==========================================
private const val BASE_URL = "http://nazariolayme.icu/api_hotel/hotel/"

// Consulta inteligente: Devuelve precio base si está libre, o precio pactado si ocupada
fun ConsultaHabitacion(num: String, resp: (Reserva?) -> Unit, ctx: Context) {
    Volley.newRequestQueue(ctx).add(JsonArrayRequest(Request.Method.GET, "${BASE_URL}consultar.php?numeroHabitacion=$num", null,
        { r ->
            if (r.length() > 0) {
                val o = r.getJSONObject(0)
                resp(Reserva(o.optInt("numeroHabitacion"), o.optString("nombreHuesped"), o.optString("fechaEntrada"),
                    o.optString("fechaSalida"), o.optDouble("precioNoche").toFloat(), o.optString("tipoHabitacion"), o.optString("estado")))
            } else resp(null)
        }, { resp(null) }))
}

fun ConsultarHabitacionesPorTipo(ctx: Context, tipo: String, resp: (List<HabitacionEstado>) -> Unit) {
    Volley.newRequestQueue(ctx).add(JsonArrayRequest(Request.Method.GET, "${BASE_URL}consultar_por_tipo.php?tipoHabitacion=$tipo", null,
        { r -> val l=mutableListOf<HabitacionEstado>(); for(i in 0 until r.length()){val o=r.getJSONObject(i); l.add(HabitacionEstado(o.optString("numeroHabitacion"), o.optString("nombreHuesped"), o.optString("estado")))}; resp(l) }, { resp(emptyList()) }))
}

fun ListarTodasLasReservas(ctx: Context, resp: (List<Reserva>) -> Unit) {
    Volley.newRequestQueue(ctx).add(JsonArrayRequest(Request.Method.GET, "${BASE_URL}listar.php", null,
        { r -> val l=mutableListOf<Reserva>(); for(i in 0 until r.length()){val o=r.getJSONObject(i); l.add(Reserva(o.optInt("numeroHabitacion"), "","", "", o.optDouble("precioNoche").toFloat(), "" ))}; resp(l) }, { resp(emptyList()) }))
}

fun GuardarReserva(r: Reserva, resp: (Boolean) -> Unit, ctx: Context) = enviarPOST("${BASE_URL}guardar.php", r, resp, ctx)
fun ModificarReserva(r: Reserva, resp: (Boolean) -> Unit, ctx: Context) = enviarPOST("${BASE_URL}modificar.php", r, resp, ctx)

fun BorrarReserva(num: String, resp: (Boolean) -> Unit, ctx: Context) {
    Volley.newRequestQueue(ctx).add(object : StringRequest(Method.POST, "${BASE_URL}borrar.php", { r -> resp(JSONObject(r).optString("resultado")=="1") }, { resp(false) }) {
        override fun getParams() = mapOf("numeroHabitacion" to num)
    })
}

fun enviarPOST(url: String, r: Reserva, resp: (Boolean) -> Unit, ctx: Context) {
    Volley.newRequestQueue(ctx).add(object : StringRequest(Method.POST, url, { re -> resp(JSONObject(re).optString("resultado")=="1") }, { resp(false) }) {
        override fun getParams() = mapOf(
            "numeroHabitacion" to r.numeroHabitacion.toString(), "nombreHuesped" to r.nombreHuesped,
            "fechaEntrada" to r.fechaEntrada, "fechaSalida" to r.fechaSalida,
            "precioNoche" to r.precioNoche.toString(), "tipoHabitacion" to r.tipoHabitacion
        )
    })
}